package gov.va.med.mhv.usermgmt.service;

import java.util.Date;

import gov.va.med.mhv.common.api.dto.PatientDTO;
import gov.va.med.mhv.common.api.exception.MHVException;
import gov.va.med.mhv.common.api.util.ResponseUtil;
import gov.va.med.mhv.common.data.model.Facility;
import gov.va.med.mhv.common.data.model.Patient;
import gov.va.med.mhv.common.data.model.UserProfile;
import gov.va.med.mhv.common.data.model.UserProfileDeactivationReason;
import gov.va.med.mhv.usermgmt.main.registry.PersonalInfo;
import gov.va.med.mhv.usermgmt.main.registry.Status;

public interface PatientService {

	public ResponseUtil<PatientDTO> getPatientByUserProfileId(Long userProfileid);

	/**
	 * Execute the UpdatePatientFacilities operation
	 *
	 * @param userName
	 * @param userProfile
	 * @param isMviEnabled
	 * @param response
	 */
	public void updatePatientFacilities(String userName, UserProfile userProfile, boolean isMviEnabled, ResponseUtil response);

	/**
	 * Starts the correlation with MPI of a given patient. Requires the patient
	 * to be uncorrelated.
	 *
	 * @param patient
	 * @param response
	 */
	public void correlate(Patient patient, ResponseUtil response);

	/**
	 * Execute the GetPatientFromMVI operation
	 *
	 * @param userProfile
	 * @param atRegistration
	 * @param response
	 */
	public void getPatientFromMVI(UserProfile userProfile, Boolean atRegistration, ResponseUtil response);

	/**
	 * Execute the GetPatientForFccx operation
	 *
	 * @param userProfile
	 * @param fccxIcn
	 * @param response
	 */
	public void createPatientForFccx(UserProfile userProfile, String fccxIcn, ResponseUtil response);

	/**
	 * Execute the GetPatientFromRegistryWithMiddleName operation
	 *
	 * @param firstName
	 * @param lastName
	 * @param middleName
	 * @param gender
	 * @param birthDate
	 * @param ssn
	 * @param userId
	 * @param response
	 */
	public void getPatientFromRegistryWithMiddleName(String firstName, String lastName, String middleName, String gender, Date birthDate, String ssn, Long userProfileId, ResponseUtil response);

	/**
	 * Execute the GetPatientFromRegistry operation
	 *
	 * @param firstName
	 * @param lastName
	 * @param birthDate
	 * @param ssn
	 * @param userId
	 * @return PatientServiceResponse
	 */
	public void getPatientFromRegistry(String firstName, String lastName, Date birthDate, String ssn, Long userProfileId, ResponseUtil response);

	/**
	 * Execute the GetPatientForUser operation
	 *
	 * @param userProfile
	 * @return PatientServiceResponse
	 */
	public void getPatientForUser(UserProfile userProfile, ResponseUtil response);

	/**
	 * Retrieve a patient for a given ICN
	 *
	 * @param icn The ICN to lookup the patient for.
	 * @return PatientServiceResponse
	 */
	public void getPatientByIcn(String icn, ResponseUtil response);

	/**
	 * Retrieve a patient for a given Partial ICN
	 *
	 * @param icn The partial ICN to lookup the patient for.
	 * @return PatientCollectionServiceResponse
	 */
	public void getPatientByPartialIcn(String icn, ResponseUtil response);

	/**
	 * Retrieves patient by id.
	 *
	 * @param id The id to lookup the patient for.
	 * @return PatientServiceResponse
	 */
	public void getPatientById(Long id, ResponseUtil response);

	/**
	 * Execute the CreatePatientForUser operation
	 *
	 * @param patient
	 * @param userProfile
	 * @return PatientServiceResponse
	 */
	public void createPatientForUser(Patient patient, UserProfile userProfile, ResponseUtil response);

	/**
	 * Execute the AddFacilityForPatient operation
	 *
	 * @param patient
	 * @param facility
	 * @return PatientServiceResponse
	 */
	public void addFacilityForPatient(Patient patient, Facility facility, ResponseUtil response);

	/**
	 * Updates a given patient.
	 * Processes and registers registry changes (ICN and/or facility changes).
	 * Only updates to the registry information, i.e ICN, patient
	 * facilities, first name, last name, date of birth and ssn.
	 * Changes to other properties are ignored.
	 * Changes are determines by comparing (based on its primary key)
	 * the given and the current patient (as persisted in the EVault).
	 *
	 * @param patient The patient to update.
	 * @return PatientServiceResponse
	 */
	public void updatePatientRegistryInformation(Patient patient, ResponseUtil response);

	/**
	 * JAZZ Defect# 20309: If there four or more traits update, DO NOT Update the user and deactivate the user.
	 *
	 * @param patient The patient to invalidate.
	 * @param referenceId Denotes a referenceId to use when refering to
					      this action.
	 * @param adminNetworkId
	 * @param reason Denotes the reason to deactivate the patient/user.
	 * @param note Note for the deactivation of the user.
	 * @return PatientServiceResponse
	 */
	public void deactivatePatientForMultiTraitUpdateError(Patient patient, String referenceId, String adminNetworkId, UserProfileDeactivationReason reason, String note, ResponseUtil response);

	/**
	 * @param patient The patient to invalidate.
	 * @param referenceId Denotes a referenceId to use when refering to
					      this action.
	 * @param adminNetworkId
	 * @param reason Denotes the reason to deactivate the patient/user.
	 * @param note Note for the deactivation of the user.
	 * @return Status
	 * @throws MHVException 
	 */
	public Status deactivatePatient(gov.va.med.mhv.common.data.model.Patient patient, String referenceId, String reason, String note) throws MHVException;
	
	/**
	 * Store Patient details in database
	 *
	 * @param patient
	 * @return patient
	 */
	public Patient savePatient(Patient patient);

	/**
	 * Store both the patient and the patient's associated facilities.
	 * @param patient The patient being stored.
	 * @return The persisted patient.
	 */
	public Patient savePatientWithFacilities(Patient patient);
	
	/**
	 * Change Patient ICN in database
	 *
	 * @param patient
	 * @param icn
	 * @return status
	 */
	public Status changeIcn(Patient patient, String icn);

	/**
	 * Change Patient ICN in database
	 *
	 * @param fromPatient
	 * @param toPatient
	 * @param referenceId
	 * @return status
	 */
	public Status mergePatients(Patient fromPatient, Patient toPatient,	String referenceId);

	/**
	 * Change update personal info in database
	 *
	 * @param patient
	 * @param personalInfo
	 * @return status
	 */
	public Status updatePersonalInfo(Patient patient, PersonalInfo personalInfo);

	//public void healthSummaryBatch();

	//public void reactivatePatientInHelpDesk(Patient patient, String adminUserName, UserProfileDeactivationReason reason, String note, ResponseUtil responseUtil);
	
}
